import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull, move_to_side, move_to_anchor, pick_drawer

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset environment & start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve object positions
        positions = get_object_positions()
        # assume positions dict contains keys:
        #   'drawer_handle_side' : 3‐vector side approach point
        #   'drawer_handle_anchor' : 3‐vector anchor grasp point
        #   'gripper_start' : current gripper position
        side_pos = positions['drawer_handle_side']
        anchor_pos = positions['drawer_handle_anchor']

        # 1) Rotate gripper to 90 degrees about its local z‐axis
        print("[Task] Rotating gripper to ninety_deg orientation")
        # quaternion for 90° about z: [x,y,z,w] = [0,0,sin(π/4),cos(π/4)]
        target_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Episode ended during rotate")
            return

        # 2) Move to side position of drawer
        print("[Task] Moving gripper to side position of drawer")
        obs, reward, done = move_to_side(env, task, side_pos)
        if done:
            print("[Task] Episode ended during move_to_side")
            return

        # 3) Move from side to anchor position to grasp handle
        print("[Task] Moving gripper to anchor position on drawer")
        obs, reward, done = move_to_anchor(env, task, anchor_pos)
        if done:
            print("[Task] Episode ended during move_to_anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick_drawer(env, task)
        if done:
            print("[Task] Episode ended during pick_drawer")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Episode ended during pull")
            return

        print("[Task] Drawer should now be open – task complete")

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()