import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset and prepare recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get object positions
        positions = get_object_positions()
        # assume we want to move 'object_1' to 'destination'
        if 'object_1' not in positions:
            raise KeyError("object_1 not found in environment")
        obj1_pos = positions['object_1']

        # choose a destination: if named 'destination' exists, use it; otherwise use next available
        if 'destination' in positions:
            dest_pos = positions['destination']
        else:
            keys = [k for k in positions.keys() if k != 'object_1']
            if not keys:
                raise KeyError("No valid destination found")
            dest_pos = positions[keys[0]]

        # Step 1: approach and pick up object_1
        print("[Task] Picking up object_1 at:", obj1_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=obj1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended prematurely after pick")
            return

        # Step 2: place object_1 at destination
        print("[Task] Placing object_1 at:", dest_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=dest_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended prematurely after place")
            return

        # (Optional) further steps could go here, e.g., rotate gripper or pull a drawer
        # Example:
        # print("[Task] Rotating gripper to 90 degrees")
        # target_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        # obs, reward, done = rotate(env, task, target_quat)
        # if done:
        #     return

        print("[Task] Completed pick-and-place sequence successfully")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()