import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset to initial state and get first observation
        descriptions, obs = task.reset()

        # Initialize video writers if desired
        init_video_writers(obs)

        # Wrap the step and observation functions for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We expect keys like 'drawer_side_handle', 'drawer_anchor_handle', etc.
        try:
            side_handle_key = [k for k in positions.keys() if 'side' in k][0]
            anchor_handle_key = [k for k in positions.keys() if 'anchor' in k][0]
        except IndexError:
            print("Error: Could not find 'side' or 'anchor' in object positions.")
            return

        side_pos = positions[side_handle_key]
        anchor_pos = positions[anchor_handle_key]
        print(f"[Task] Side handle position: {side_pos}")
        print(f"[Task] Anchor handle position: {anchor_pos}")

        # === Oracle Plan Execution ===

        # 1) Rotate gripper to 90 degrees around Y (example quaternion)
        ninety_deg_quat = np.array([0.0, 0.7071, 0.0, 0.7071])
        print("[Task] Rotating gripper to target orientation:", ninety_deg_quat)
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Terminated during rotation.")
            return

        # 2) Move to side position of the drawer handle
        print("[Task] Moving to side position")
        obs, reward, done = move(
            env, task,
            target_pos=side_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during move-to-side.")
            return

        # 3) Move to anchor position of the drawer handle
        print("[Task] Moving to anchor position")
        obs, reward, done = move(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated during move-to-anchor.")
            return

        # 4) Pick (attach to) the drawer handle
        print("[Task] Picking drawer handle")
        obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.0,
            max_steps=50,
            threshold=0.001,
            approach_axis='x',
            timeout=5.0
        )
        if done:
            print("[Task] Terminated during pick-handle.")
            return

        # 5) Pull drawer open by moving end-effector along +Z
        pull_target = (anchor_pos[0], anchor_pos[1], anchor_pos[2] + 0.2)
        print("[Task] Pulling drawer to:", pull_target)
        obs, reward, done = pull(
            env, task,
            target_pos=pull_target,
            max_steps=50,
            threshold=0.01,
            approach_axis='z',
            timeout=5.0
        )
        if done:
            print("[Task] Terminated during pull.")
            return

        print("[Task] Drawer should now be open.")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()