import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        if not positions:
            print("[Error] No object positions returned; aborting task.")
            return

        # === Exploration Phase: discover properties of all objects ===
        print("[Phase] Starting exploration of all objects to identify missing predicates")
        # assume keys for objects start with 'object'
        object_keys = [k for k in positions.keys() if k.startswith('object')]
        for obj_name in object_keys:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Moving to {obj_name} at {obj_pos}")
            try:
                obs, reward, done = move(
                    env, task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    timeout=5.0
                )
                if done:
                    print(f"[Exploration] Task ended unexpectedly during move to {obj_name}")
                    return
                print(f"[Exploration] Picking up {obj_name}")
                obs, reward, done = pick(
                    env, task,
                    target_pos=obj_pos,
                    approach_distance=0.05,
                    max_steps=100,
                    threshold=0.01,
                    timeout=5.0
                )
                if done:
                    print(f"[Exploration] Task ended unexpectedly during pick of {obj_name}")
                    return
                # place back in roughly same spot but slightly offset in Z
                place_pos = obj_pos + np.array([0.0, 0.0, 0.10])
                print(f"[Exploration] Placing {obj_name} at {place_pos}")
                obs, reward, done = place(
                    env, task,
                    target_pos=place_pos,
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.01,
                    timeout=5.0
                )
                if done:
                    print(f"[Exploration] Task ended unexpectedly during place of {obj_name}")
                    return
            except Exception as e:
                print(f"[Warning] Exception while exploring {obj_name}: {e}")
        print("[Phase] Completed object exploration")

        # === Drawer Lock Exploration: identify lock-known predicate ===
        if 'drawer_handle' in positions:
            handle_pos = positions['drawer_handle']
            print("[Phase] Exploring drawer handle at", handle_pos)
            try:
                obs, reward, done = move(
                    env, task,
                    target_pos=handle_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    timeout=5.0
                )
                if done:
                    print("[Phase] Task ended during move to drawer handle")
                    return
                print("[Phase] Grasping drawer handle")
                obs, reward, done = pick(
                    env, task,
                    target_pos=handle_pos,
                    approach_distance=0.02,
                    max_steps=100,
                    threshold=0.01,
                    timeout=5.0
                )
                if done:
                    print("[Phase] Task ended during pick-drawer-handle")
                    return
                print("[Phase] Pulling drawer to learn lock status")
                obs, reward, done = pull(env, task)
                if done:
                    print("[Phase] Task ended during pull")
                    return
                print("[Phase] Drawer lock status exploration complete")
            except Exception as e:
                print(f"[Warning] Exception during drawer exploration: {e}")
        else:
            print("[Info] No 'drawer_handle' key found in positions, skipping drawer exploration")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()