import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Identify the side and anchor positions for the drawer handle
        # Try a few common keys; adapt if your environment uses different names
        if 'drawer_side_pos' in positions:
            side_pos = positions['drawer_side_pos']
        elif 'drawer_side' in positions:
            side_pos = positions['drawer_side']
        else:
            raise KeyError("Cannot find drawer side position in object_positions output")

        if 'drawer_anchor_pos' in positions:
            anchor_pos = positions['drawer_anchor_pos']
        elif 'drawer_anchor' in positions:
            anchor_pos = positions['drawer_anchor']
        else:
            raise KeyError("Cannot find drawer anchor position in object_positions output")

        # === Oracle Plan Execution ===

        # 1) Ensure gripper is rotated from zero_deg to ninety_deg
        print("[Task] Rotating gripper from zero_deg to ninety_deg")
        try:
            obs, reward, done = rotate(env, task, 'gripper', 'zero_deg', 'ninety_deg')
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            return
        if done:
            print("[Task] Finished during rotate")
            return

        # 2) Move to drawer side position
        print(f"[Task] Moving to drawer side position: {side_pos}")
        try:
            obs, reward, done = move(env, task, target_pos=np.array(side_pos))
        except Exception as e:
            print(f"[Error] move to side failed: {e}")
            return
        if done:
            print("[Task] Finished during move to side")
            return

        # 3) Move to drawer anchor position (handle)
        print(f"[Task] Moving to drawer anchor position: {anchor_pos}")
        try:
            obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
        except Exception as e:
            print(f"[Error] move to anchor failed: {e}")
            return
        if done:
            print("[Task] Finished during move to anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Picking drawer handle")
        try:
            obs, reward, done = pick(env, task, target_pos=np.array(anchor_pos))
        except Exception as e:
            print(f"[Error] pick handle failed: {e}")
            return
        if done:
            print("[Task] Finished during pick")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling drawer open")
        try:
            obs, reward, done = pull(env, task)
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            return
        if done:
            print("[Task] Finished during pull")
            return

        print("[Task] Oracle plan completed successfully!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()