import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation to record video
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve object and position dictionary
        positions = get_object_positions()
        # assume get_object_positions returns keys:
        # 'drawer_side_pos', 'drawer_anchor_pos'
        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            raise KeyError("Expected keys 'drawer_side_pos' and 'drawer_anchor_pos' in object_positions")

        drawer_side_pos = np.array(positions['drawer_side_pos'])
        drawer_anchor_pos = np.array(positions['drawer_anchor_pos'])

        # 1) rotate gripper to ninety_deg
        print("[Task] Rotating gripper from zero_deg to ninety_deg")
        obs, reward, done = rotate(env, task, 'gripper', 'zero_deg', 'ninety_deg')
        if done:
            print("[Task] Task ended after rotate")
            return

        # 2) move to side position of drawer
        print("[Task] Moving gripper to side position:", drawer_side_pos)
        obs, reward, done = move(env, task, target_pos=drawer_side_pos)
        if done:
            print("[Task] Task ended after moving to side")
            return

        # 3) move to anchor position (the handle)
        print("[Task] Moving gripper to anchor position:", drawer_anchor_pos)
        obs, reward, done = move(env, task, target_pos=drawer_anchor_pos)
        if done:
            print("[Task] Task ended after moving to anchor")
            return

        # 4) pick the drawer handle
        print("[Task] Grasping drawer handle at:", drawer_anchor_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.005,
            approach_axis='-z',
            timeout=5.0
        )
        if done:
            print("[Task] Task ended during pick")
            return

        # 5) pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended after pull")
            return

        print("[Task] Drawer should now be open. Task complete.")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()