import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset to initial state and set up recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We expect at least two labelled positions for the drawer: side and anchor
        side_key = next((k for k in positions if "side" in k), None)
        anchor_key = next((k for k in positions if "anchor" in k), None)
        if side_key is None or anchor_key is None:
            raise KeyError("Could not find 'side' or 'anchor' positions in object_positions")

        side_pos = positions[side_key]
        anchor_pos = positions[anchor_key]

        # Predefined quaternions for gripper orientation
        zero_quat = [0.0, 0.0, 0.0, 1.0]
        ninety_quat = [0.7071, 0.0, 0.0, 0.7071]

        # === Exploration Phase ===
        # Move to anchor position to grab the handle and test whether the drawer is locked
        print("[Exploration] Moving to handle anchor position")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Exploration] Task ended prematurely after move to anchor.")
            return

        print("[Exploration] Grasping handle to test lock state")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Exploration] Task ended prematurely after pick.")
            return

        # Attempt to pull once purely to learn lock state
        print("[Exploration] Attempting pull to identify lock state")
        try:
            obs, reward, done = pull(env, task)
            if done:
                print("[Exploration] Task ended after pull.")
                return
            print("[Exploration] Pull succeeded or drawer unlocked.")
        except Exception as e:
            # We assume a failure indicates the drawer is locked
            print(f"[Exploration] Pull failed, drawer appears locked: {e}")

        # Always place the handle back so we can re‐grasp after rotation
        print("[Exploration] Releasing handle")
        obs, reward, done = place(env, task, target_pos=anchor_pos)
        if done:
            print("[Exploration] Task ended prematurely after place.")
            return

        # === Oracle Plan Execution ===
        # 1) Rotate gripper to zero orientation (safe home)
        print("[Plan] Rotating gripper to zero orientation")
        obs, reward, done = rotate(env, task, zero_quat)
        if done:
            print("[Plan] Task ended during rotate to zero.")
            return

        # 2) Rotate gripper to ninety degrees to align with drawer handle
        print("[Plan] Rotating gripper to ninety degrees")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Plan] Task ended during rotate to ninety.")
            return

        # 3) Move to side position of the drawer
        print("[Plan] Moving to side position")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Plan] Task ended after move to side.")
            return

        # 4) Move to anchor position again (now with correct orientation)
        print("[Plan] Moving to anchor position")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Task ended after move to anchor.")
            return

        # 5) Grasp the drawer handle
        print("[Plan] Grasping drawer handle")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Task ended after pick handle.")
            return

        # 6) Pull to open the drawer
        print("[Plan] Pulling to open drawer")
        obs, reward, done = pull(env, task)
        if done:
            print("[Plan] Task ended after pull.")
            return

        print("[Plan] Drawer should now be open. Task complete.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()