import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        positions = get_object_positions()
        # Define named positions and objects from the dictionary
        # These keys must match what get_object_positions returns
        drawer_handle_pos = positions['drawer_handle']
        drawer_side_pos   = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']
        inside_obj_pos    = positions['inside_object']
        target_bin_pos    = positions['target_bin']
        ninety_deg_quat   = [0.707, 0, 0, 0.707]  # example target orientation for opening

        # === Plan Execution ===
        # 1) Rotate gripper to align with drawer handle
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, ninety_deg_quat, max_steps=100, threshold=0.02, timeout=5.0)
        if done:
            print("[Task] Ended prematurely during rotate")
            return

        # 2) Move close to the side position of the drawer
        print("[Task] Moving to drawer side position at", drawer_side_pos)
        obs, reward, done = move(env, task,
                                 target_pos=drawer_side_pos,
                                 approach_distance=0.04,
                                 max_steps=100,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Task] Ended prematurely during move to side")
            return

        # 3) Move close to the anchor position of the drawer
        print("[Task] Moving to drawer anchor position at", drawer_anchor_pos)
        obs, reward, done = move(env, task,
                                 target_pos=drawer_anchor_pos,
                                 approach_distance=0.02,
                                 max_steps=100,
                                 threshold=0.005,
                                 timeout=10.0)
        if done:
            print("[Task] Ended prematurely during move to anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Picking drawer handle at", drawer_handle_pos)
        obs, reward, done = pick(env, task,
                                 target_pos=drawer_handle_pos,
                                 approach_distance=0.01,
                                 max_steps=80,
                                 threshold=0.005,
                                 approach_axis='z',
                                 timeout=5.0)
        if done:
            print("[Task] Ended prematurely during pick(handle)")
            return

        # 5) Pull the drawer to open it
        print("[Task] Pulling drawer")
        obs, reward, done = pull(env, task,
                                 target_pos=drawer_anchor_pos,
                                 max_steps=60,
                                 timeout=5.0)
        if done:
            print("[Task] Ended prematurely during pull")
            return

        # 6) Move inside the drawer to reach the object
        print("[Task] Moving inside drawer to object at", inside_obj_pos)
        obs, reward, done = move(env, task,
                                 target_pos=inside_obj_pos,
                                 approach_distance=0.03,
                                 max_steps=100,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Task] Ended prematurely moving inside drawer")
            return

        # 7) Pick the object inside drawer
        print("[Task] Picking object inside drawer")
        obs, reward, done = pick(env, task,
                                 target_pos=inside_obj_pos,
                                 approach_distance=0.015,
                                 max_steps=80,
                                 threshold=0.005,
                                 approach_axis='z',
                                 timeout=5.0)
        if done:
            print("[Task] Ended prematurely during pick(object)")
            return

        # 8) Move to target bin location
        print("[Task] Moving to target bin at", target_bin_pos)
        obs, reward, done = move(env, task,
                                 target_pos=target_bin_pos,
                                 approach_distance=0.05,
                                 max_steps=120,
                                 threshold=0.01,
                                 timeout=15.0)
        if done:
            print("[Task] Ended prematurely moving to bin")
            return

        # 9) Place the object into the bin
        print("[Task] Placing object into bin")
        obs, reward, done = place(env, task,
                                  target_pos=target_bin_pos,
                                  approach_distance=0.02,
                                  max_steps=80,
                                  threshold=0.005,
                                  approach_axis='z',
                                  timeout=5.0)
        if done:
            print("[Task] Ended prematurely during place(object)")
            return

        print("===== Task Completed Successfully =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()