import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull  # use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset task to initial state
        descriptions, obs = task.reset()

        # Initialize video recording if desired
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        object_keys = list(positions.keys())
        if len(object_keys) < 2:
            raise RuntimeError("Not enough objects in the scene to run pick‐and‐place.")

        # Select two objects: one to pick, one to serve as placement target
        source_name = object_keys[0]
        target_name = object_keys[1]
        source_pos = np.array(positions[source_name])
        target_pos = np.array(positions[target_name])

        # === Execute Oracle Plan ===
        # 1) Pick up the first object
        print(f"[Task] Picking object '{source_name}' at position {source_pos}")
        obs, reward, done = pick(
            env,
            task,
            target_pos=source_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended right after pick.")
            return

        # 2) Place the object at the second object's location
        print(f"[Task] Placing '{source_name}' at location of '{target_name}': {target_pos}")
        obs, reward, done = place(
            env,
            task,
            target_pos=target_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended right after place.")
            return

        print("[Task] Pick-and-place sequence completed successfully.")

    except Exception as e:
        print(f"[Task] Exception during execution: {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
