import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve the current gripper or robot location if available
        # assume a default “home” or the initial robot base location
        # for simplicity we won’t explicitly track base location here

        # get object positions from the scene
        positions = get_object_positions()
        # this should return a dict: e.g. {'tomato1': (x,y,z), 'tomato2': (...), 'plate': (...)}

        # we expect at least one tomato and a plate
        if 'tomato1' not in positions or 'plate' not in positions:
            raise KeyError("Required objects not found in positions: 'tomato1' and 'plate'")

        plate_pos = np.array(positions['plate'])
        tomatoes = [k for k in positions.keys() if k.startswith('tomato')]
        tomatoes.sort()

        for idx, tom_key in enumerate(tomatoes, start=1):
            tom_pos = np.array(positions[tom_key])
            print(f"[Task] Processing {tom_key} at {tom_pos}")

            # move to the tomato
            print(f"[Task] Moving towards {tom_key}")
            try:
                obs, reward, done = move(env, task, target_pos=tom_pos)
            except Exception as e:
                print(f"[Error] move failed for {tom_key}: {e}")
                return
            if done:
                print("[Task] ended during move")
                return

            # pick the tomato
            print(f"[Task] Picking up {tom_key}")
            try:
                obs, reward, done = pick(env, task, target_pos=tom_pos)
            except Exception as e:
                print(f"[Error] pick failed for {tom_key}: {e}")
                return
            if done:
                print("[Task] ended during pick")
                return

            # move to the plate
            print(f"[Task] Moving to plate at {plate_pos}")
            try:
                obs, reward, done = move(env, task, target_pos=plate_pos)
            except Exception as e:
                print(f"[Error] move to plate failed: {e}")
                return
            if done:
                print("[Task] ended during move to plate")
                return

            # place the tomato on the plate
            print(f"[Task] Placing {tom_key} on plate")
            try:
                obs, reward, done = place(env, task, target_pos=plate_pos)
            except Exception as e:
                print(f"[Error] place failed for {tom_key}: {e}")
                return
            if done:
                print("[Task] ended during place")
                return

            print(f"[Task] {tom_key} placed successfully")

        print("[Task] All tomatoes placed on plate. Task completed successfully.")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()