import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve object positions
        positions = get_object_positions()
        # These keys must match those returned by get_object_positions()
        drawer_side_pos = positions["drawer_side_pos"]
        drawer_anchor_pos = positions["drawer_anchor_pos"]
        # Name of the gripper as expected by the skill implementations
        gripper_name = "gripper"

        # === Oracle Plan Execution ===

        # 1) Rotate the gripper from its initial angle to ninety_deg
        print("[Plan] Rotating gripper to ninety_deg")
        try:
            obs, reward, done = rotate(env, task, gripper_name, "zero_deg", "ninety_deg")
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            return
        if done:
            print("[Plan] Task ended after rotate")
            return

        # 2) Move gripper to the side of the drawer
        print("[Plan] Moving gripper to side position", drawer_side_pos)
        try:
            obs, reward, done = move(env, task, gripper_name, drawer_side_pos)
        except Exception as e:
            print(f"[Error] move to side failed: {e}")
            return
        if done:
            print("[Plan] Task ended after move-to-side")
            return

        # 3) Move gripper to the anchor position on the drawer handle
        print("[Plan] Moving gripper to anchor position", drawer_anchor_pos)
        try:
            obs, reward, done = move(env, task, gripper_name, drawer_anchor_pos)
        except Exception as e:
            print(f"[Error] move to anchor failed: {e}")
            return
        if done:
            print("[Plan] Task ended after move-to-anchor")
            return

        # 4) Pick the drawer handle (approach and grasp)
        print("[Plan] Picking drawer handle at", drawer_anchor_pos)
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=drawer_anchor_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis="z",
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] pick failed: {e}")
            return
        if done:
            print("[Plan] Task ended after pick-drawer")
            return

        # 5) Pull the drawer open
        print("[Plan] Pulling drawer")
        try:
            obs, reward, done = pull(env, task, gripper_name, "drawer")
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            return
        if done:
            print("[Plan] Task ended after pull")
            return

        print("[Plan] Drawer should now be open. Task completed successfully.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()