import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Info] Retrieved object positions:", positions.keys())

        # Check for required objects, warn if missing
        required = [
            "bottom_anchor_pos", "middle_anchor_pos", "top_anchor_pos",
            "rubbish", "drawer_handle_pos", "bin"
        ]
        missing = [k for k in required if k not in positions]
        if missing:
            print(f"[Warning] Missing positions for objects: {missing}")
        # we proceed but will guard missing entries when used

        # === Exploration Phase ===
        print("===== Exploration Phase: Identify Objects and Learn Attributes =====")
        robot_loc = None
        try:
            # assume initial robot location at first gripper pose
            obs0 = task.get_observation()
            robot_loc = obs0.gripper_pose[:3]
        except Exception as e:
            print("[Warning] Could not read initial robot pose:", e)

        for name, pos in positions.items():
            if robot_loc is None or pos is None:
                continue
            try:
                print(f"[Exploration] Moving to {name} at {pos}")
                obs, reward, done = move(env, task, robot_loc, pos)
                robot_loc = pos
                print(f"[Exploration] Identified {name}")
                if done:
                    print("[Exploration] Task ended early during identification.")
                    return
            except Exception as e:
                print(f"[Exploration] Error while identifying {name}: {e}")

        # Based on domain analysis, the missing predicate is "lock-known"
        print("[Exploration] Determined missing predicate: 'lock-known'")

        # === Plan Execution ===

        # 1) Open the drawer
        print("===== Plan Step 1: Opening Drawer =====")
        try:
            # Rotate gripper from zero to ninety degrees
            obs, reward, done = rotate(env, task, "gripper", "zero_deg", "ninety_deg")
            if done:
                print("[Plan] Task ended after rotate.")
                return
        except Exception as e:
            print("[Plan] Rotate failed:", e)

        # 2) Approach the drawer handle
        handle_pos = positions.get("drawer_handle_pos")
        if handle_pos is not None:
            try:
                print("[Plan] Picking drawer handle at:", handle_pos)
                obs, reward, done = pick(
                    env, task,
                    target_pos=np.array(handle_pos),
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Plan] Task ended during drawer-handle pick.")
                    return
            except Exception as e:
                print("[Plan] Pick drawer handle failed:", e)
        else:
            print("[Plan] Skipping drawer handle pick: position unknown")

        # 3) Pull the drawer to learn lock status and open it
        print("[Plan] Pulling drawer to open and learn lock status")
        try:
            obs, reward, done = pull(env, task, "robot", "drawer_handle", None)
            if done:
                print("[Plan] Task ended during pull.")
                return
        except Exception as e:
            print("[Plan] Pull failed:", e)

        # 4) Retrieve the rubbish
        print("===== Plan Step 2: Retrieving Rubbish =====")
        rubbish_pos = positions.get("rubbish")
        if rubbish_pos is not None:
            try:
                print("[Plan] Picking rubbish at:", rubbish_pos)
                obs, reward, done = pick(
                    env, task,
                    target_pos=np.array(rubbish_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Plan] Task ended during rubbish pick.")
                    return
            except Exception as e:
                print("[Plan] Pick rubbish failed:", e)
        else:
            print("[Plan] Skipping rubbish pick: position unknown")

        # 5) Place the rubbish in the bin
        print("===== Plan Step 3: Placing Rubbish in Bin =====")
        bin_pos = positions.get("bin")
        if bin_pos is not None:
            try:
                print("[Plan] Placing rubbish into bin at:", bin_pos)
                obs, reward, done = place(env, task, target_pos=np.array(bin_pos))
                if done:
                    print("[Plan] Task ended during place.")
                    return
            except Exception as e:
                print("[Plan] Place failed:", e)
        else:
            print("[Plan] Skipping place: bin position unknown")

        print("===== Plan Completed Successfully =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()