import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        if not positions:
            print("[Error] No objects found in the scene.")
            return

        # Replace placeholder keys with your actual object names
        # Here we pick the first two objects in the dictionary for demonstration
        obj_names = list(positions.keys())
        if len(obj_names) < 2:
            print("[Error] Need at least two objects to demonstrate pick and place.")
            return

        object_1 = obj_names[0]
        object_2 = obj_names[1]
        object_1_pos = np.array(positions[object_1])
        object_2_pos = np.array(positions[object_2])

        # === Task Plan ===
        # 1) Pick up object_1
        print(f"[Task] Manipulating {object_1} at: {object_1_pos}")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=object_1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] pick failed on {object_1}: {e}")
            return

        if done:
            print(f"[Task] Task ended after pick of {object_1}.")
            return

        # 2) Place object_1 at the location of object_2
        print(f"[Task] Placing {object_1} at the location of {object_2}: {object_2_pos}")
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=object_2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] place failed for {object_1}: {e}")
            return

        if done:
            print(f"[Task] Task ended after placing {object_1}.")
            return

        # 3) Pick up object_2
        print(f"[Task] Manipulating {object_2} at: {object_2_pos}")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=object_2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] pick failed on {object_2}: {e}")
            return

        if done:
            print(f"[Task] Task ended after pick of {object_2}.")
            return

        # 4) Place object_2 back at original position of object_1
        print(f"[Task] Placing {object_2} back at original {object_1} position: {object_1_pos}")
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=object_1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] place failed for {object_2}: {e}")
            return

        if done:
            print(f"[Task] Task ended after placing {object_2}.")
            return

        print("[Task] Completed pick-and-place sequence successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()