import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Replace the keys below with the actual names in your environment
        object_1_pos = positions['object_1']
        object_2_pos = positions['object_2']

        # Convert to numpy arrays
        object_1_pos = np.array(object_1_pos)
        object_2_pos = np.array(object_2_pos)

        # === Execute Oracle Plan ===

        # 1) Move above object_1
        print("[Task] Moving to object_1 at:", object_1_pos)
        obs, reward, done = move(
            env, task,
            target_pos=object_1_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Ended early after move to object_1.")
            return

        # 2) Pick object_1
        print("[Task] Picking object_1")
        obs, reward, done = pick(
            env, task,
            target_pos=object_1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Ended early after pick.")
            return

        # 3) Move above object_2
        print("[Task] Moving to placement location at:", object_2_pos)
        obs, reward, done = move(
            env, task,
            target_pos=object_2_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Ended early after move to object_2.")
            return

        # 4) Place object_1 at object_2 location
        print("[Task] Placing object_1 at object_2 location")
        obs, reward, done = place(
            env, task,
            target_pos=object_2_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Ended early after place.")
            return

        print("===== Task Completed Successfully =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
