# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Replace these keys with the actual names in your environment
        drawer_handle_pos = positions['drawer_handle']
        target_object_pos = positions['target_object']
        drop_zone_pos = positions['drop_zone']

        # === Plan Execution ===

        # 1) Pick the drawer handle
        print("[Task] Grasping drawer handle at:", drawer_handle_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=drawer_handle_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Finished early after pick(handle).")
            return

        # 2) Pull the drawer open
        print("[Task] Pulling the drawer open...")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Finished early after pull.")
            return

        # 3) Pick the target object inside the drawer
        print("[Task] Picking target object at:", target_object_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=target_object_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Finished early after pick(object).")
            return

        # 4) Place the object at the designated drop zone
        print("[Task] Placing object at:", drop_zone_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=drop_zone_pos
        )
        if done:
            print("[Task] Finished early after place.")
            return

        print("[Task] Plan completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
