import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap task.step and task.get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve all known object positions
        positions = get_object_positions()
        # ensure target objects exist
        if 'tomato' not in positions:
            print("Error: 'tomato' not found in environment object positions.")
            return
        if 'container' not in positions:
            print("Error: 'container' not found in environment object positions.")
            return

        tomato_pos = positions['tomato']
        container_pos = positions['container']
        print(f"[Task] Tomato position: {tomato_pos}, Container position: {container_pos}")

        # 1) Pick up the tomato
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"Exception during pick: {e}")
            return
        if done:
            print("[Task] Task ended prematurely after pick.")
            return

        # 2) Safety check before rotation
        obs = task.get_observation()
        gripper_z = obs.gripper_pose[2]
        safe_height = 0.10
        if gripper_z < safe_height:
            print(f"Error: Gripper too low for rotation (z={gripper_z:.3f}), aborting.")
            # place back and exit
            place(env, task, target_pos=tomato_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
            return

        # 3) Rotate the tomato in gripper
        # define a target orientation quaternion (xyzw)
        target_quat = np.array([0.0, 0.0, 0.7071, 0.7071])
        try:
            obs, reward, done = rotate(
                env,
                task,
                target_quat=target_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
        except Exception as e:
            print(f"Exception during rotate: {e}")
            # attempt to recover by placing back
            place(env, task, target_pos=tomato_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
            return
        if done:
            print("[Task] Task ended prematurely during rotate.")
            return

        # 4) Place the tomato into the container
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=container_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"Exception during place: {e}")
            return
        if done:
            print("[Task] Task ended prematurely after place.")
            return

        print("===== Task completed successfully! =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()