import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        if not positions:
            print("[Error] No objects found in the environment.")
            return

        # Select two objects for a simple pick-and-place plan
        obj_keys = list(positions.keys())
        if len(obj_keys) < 2:
            print("[Error] Need at least two objects to perform pick-and-place.")
            return
        source_obj = obj_keys[0]
        target_obj = obj_keys[1]
        source_pos = positions[source_obj]
        target_pos = positions[target_obj]

        # === Oracle Plan Execution ===
        # Step 1: Pick up the first object
        print(f"[Plan] Picking up '{source_obj}' at {source_pos}")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=source_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Exception during pick of '{source_obj}': {e}")
            return
        if done:
            print("[Task] Episode ended early during pick.")
            return

        # Step 2: Place the object at the target location
        print(f"[Plan] Placing '{source_obj}' at {target_pos}")
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=target_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Exception during place of '{source_obj}': {e}")
            return
        if done:
            print("[Task] Episode ended early during place.")
            return

        print("[Task] Pick-and-place plan executed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()