import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset environment
        descriptions, obs = task.reset()

        # video recording wrappers
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get all object positions from the scene
        positions = get_object_positions()

        # identify the side position key for the drawer handle
        side_keys = [k for k in positions if 'side' in k]
        if not side_keys:
            raise RuntimeError("Missing side position in object_positions. (is-anchor-pos)")
        side_key = side_keys[0]
        side_pos = np.array(positions[side_key])

        # identify the anchor position key for the drawer handle
        anchor_keys = [k for k in positions if 'anchor' in k]
        if not anchor_keys:
            raise RuntimeError("Missing anchor position in object_positions. (is-anchor-pos)")
        anchor_key = anchor_keys[0]
        anchor_pos = np.array(positions[anchor_key])

        # === Oracle plan steps ===

        # 1) Rotate gripper from zero_deg to ninety_deg
        print(f"[Plan] Rotating gripper from zero_deg to ninety_deg")
        obs, reward, done = rotate(env, task, 'zero_deg', 'ninety_deg')
        if done:
            print("[Plan] Task ended after rotate")
            return

        # 2) Move to side position of drawer handle
        print(f"[Plan] Moving to side position '{side_key}': {side_pos}")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Plan] Task ended after move to side")
            return

        # 3) Move to anchor position of drawer handle
        print(f"[Plan] Moving to anchor position '{anchor_key}': {anchor_pos}")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] Task ended after move to anchor")
            return

        # 4) Pick the drawer handle at anchor position
        print(f"[Plan] Picking drawer handle at anchor position")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended after pick")
            return

        # 5) Pull the drawer
        print("[Plan] Pulling the drawer")
        obs, reward, done = pull(env, task)
        if done:
            print("[Plan] Task ended after pull")
            return

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()