import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset to initial state and get PDDL descriptions
        descriptions, obs = task.reset()
        init_desc = descriptions[1] if len(descriptions) > 1 else descriptions[0]
        print("[Info] Problem init description:", init_desc)

        # Initialize video recording
        init_video_writers(obs)
        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Exploration Phase: Identify missing predicates ===
        # We expect certain predicates in the init, e.g. (handempty)
        expected_preds = ['handempty']
        missing = [p for p in expected_preds if f"({p})" not in init_desc]
        if missing:
            print(f"[Exploration] Missing predicates in init: {missing}")
            # Cannot proceed with plan until the init is corrected
            return
        else:
            print("[Exploration] All expected predicates present in init.")

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Info] Retrieved object positions:", positions)

        # === Task Plan Execution ===
        # For each object in the environment, approach, pick it up, and place it back
        for obj_name, obj_pos in positions.items():
            print(f"[Task] Approaching {obj_name} at position {obj_pos}")
            try:
                # Move to the object
                obs, reward, done = move(
                    env, task,
                    target_pos=obj_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Task] Episode ended during move. Exiting.")
                    return

                # Pick the object
                obs, reward, done = pick(
                    env, task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Task] Episode ended after picking {obj_name}. Exiting.")
                    return

                # Place the object back at the same position
                obs, reward, done = place(
                    env, task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Task] Episode ended after placing {obj_name}. Exiting.")
                    return

                print(f"[Task] Successfully manipulated {obj_name}")

            except Exception as e:
                print(f"[Error] Exception while handling {obj_name}: {e}")
                # Continue with next object
                continue

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()