import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step functions so we record observations
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get a dictionary of named positions in the scene
        positions = get_object_positions()
        # we expect at least:
        #   'drawer_side_pos' : side‐facing pose of the drawer handle
        #   'drawer_anchor_pos' : close‐in pose on the drawer handle
        #   'ninety_deg_quat'   : quaternion representing a 90° rotation
        # you may need to inspect get_object_positions() to confirm the exact keys

        side_pos = positions['drawer_side_pos']
        anchor_pos = positions['drawer_anchor_pos']
        ninety_quat = positions.get('ninety_deg_quat', None)
        if ninety_quat is None:
            # fallback: z‑axis 90° rotation quaternion
            ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])

        # STEP 1: rotate gripper to 90° so it is aligned to open the drawer
        print("[Plan] STEP 1: rotate to 90°")
        obs, reward, done = rotate(env, task, ninety_quat, max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Plan] Task ended unexpectedly during rotate")
            return

        # STEP 2: move into side position of drawer
        print("[Plan] STEP 2: approach drawer side position")
        obs, reward, done = move(
            env,
            task,
            target_pos=side_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended unexpectedly during move to side")
            return

        # STEP 3: move into anchor (grasp) position on the drawer handle
        print("[Plan] STEP 3: approach drawer handle anchor position")
        obs, reward, done = move(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended unexpectedly during move to anchor")
            return

        # STEP 4: grasp the drawer handle (pick)
        print("[Plan] STEP 4: pick the drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.0,
            max_steps=50,
            threshold=0.005,
            approach_axis='z',
            timeout=5.0
        )
        if done:
            print("[Plan] Task ended unexpectedly during pick")
            return

        # STEP 5: pull to open the drawer
        print("[Plan] STEP 5: pull to open drawer")
        obs, reward, done = pull(
            env,
            task,
            max_steps=100,
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended after pull")
            return

        print("===== Plan executed successfully! Drawer should now be open. =====")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()