# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if desired
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        if not positions:
            print("[Task] No objects found in the scene.")
            return

        # === Exploration Phase: Identify Missing Predicate ===
        missing_predicate = None

        # 1) Explore each object by moving and picking it up
        for name, pos in positions.items():
            print(f"[Explore] Moving to '{name}' at position {pos}")
            obs, reward, done = move(env, task, target_pos=pos)
            if done:
                print("[Explore] Task ended during move.")
                return
            print(f"[Explore] Attempting to pick '{name}'")
            try:
                obs, reward, done = pick(env, task, target_pos=pos)
            except Exception as e:
                err = str(e)
                if "known" in err or "predicate" in err:
                    missing_predicate = err
                    print(f"[Explore] Caught exception indicating missing predicate during pick: {err}")
                    break
                else:
                    print(f"[Explore] Unexpected exception: {err}")
                    raise
            if done:
                print("[Explore] Task ended during pick.")
                return

            # Place it back to avoid side‐effects
            print(f"[Explore] Placing '{name}' back")
            obs, reward, done = place(env, task, target_pos=pos)
            if done:
                print("[Explore] Task ended during place.")
                return

        # 2) Explore drawers (if any) by rotating and pulling to detect lock status
        if missing_predicate is None:
            for name, pos in positions.items():
                if "drawer" not in name.lower():
                    continue
                print(f"[Explore] Rotating gripper for drawer '{name}'")
                # assume rotate signature: rotate(env, task, from_angle, to_angle)
                try:
                    obs, reward, done = rotate(env, task, "zero_deg", "ninety_deg")
                except Exception as e:
                    err = str(e)
                    if "predicate" in err or "known" in err:
                        missing_predicate = err
                        print(f"[Explore] Missing predicate indicated during rotate: {err}")
                        break
                    else:
                        raise
                if done:
                    print("[Explore] Task ended during rotate.")
                    return

                print(f"[Explore] Pulling drawer '{name}' to test lock")
                try:
                    obs, reward, done = pull(env, task)
                except Exception as e:
                    err = str(e)
                    if "lock-known" in err or "predicate" in err:
                        missing_predicate = "lock-known"
                        print(f"[Explore] Identified missing predicate: lock-known")
                        break
                    else:
                        print(f"[Explore] Unexpected exception during pull: {err}")
                        raise
                if done:
                    print("[Explore] Task ended during pull.")
                    return

        # Report findings
        if missing_predicate:
            print(f"[Task] Exploration complete. Missing predicate discovered: {missing_predicate}")
        else:
            print("[Task] Exploration complete. No missing predicate detected.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()