import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Replace the placeholder keys with actual names from your environment
        # For the drawer opening task we expect keys like:
        #   'drawer_side_pos', 'drawer_anchor_pos', maybe 'home_pos'
        try:
            side_pos = np.array(positions['drawer_side_pos'])
            anchor_pos = np.array(positions['drawer_anchor_pos'])
            home_pos = np.array(positions.get('home_pos', [0.0, 0.0, 0.5]))
        except KeyError as e:
            print(f"[Error] Missing position key in object_positions: {e}")
            return

        # === Exploration Phase ===
        # We attempt a pull to surface any missing predicate (e.g., lock-known).
        print("[Exploration] Attempting exploratory pull to detect lock status predicate...")
        try:
            # We call pull even though drawer may be locked; this will raise if predicate is missing
            obs, reward, done = pull(env, task,
                                     target_pos=anchor_pos,
                                     approach_distance=0.10,
                                     max_steps=50,
                                     threshold=0.01,
                                     approach_axis='z',
                                     timeout=5.0)
            print("[Exploration] Pull succeeded or returned normally.")
        except Exception as e:
            print(f"[Exploration] Detected missing condition or failure in pull(): {e}")
            print("[Exploration] In PDDL terms, the 'lock-known' predicate may be missing.")

        # === Task Plan: Open the Drawer ===
        # 1) Rotate gripper to ninety_deg
        print("[Task] Rotating gripper to ninety_deg")
        try:
            obs, reward, done = rotate(env, task,
                                       target_angle='ninety_deg',
                                       max_steps=50,
                                       timeout=5.0)
            if done:
                print("[Task] Task ended during rotate.")
                return
        except Exception as e:
            print(f"[Error] rotate() failed: {e}")
            return

        # 2) Move from home to side position of drawer
        print(f"[Task] Moving gripper from home to side position {side_pos}")
        try:
            obs, reward, done = move(env, task,
                                     target_pos=side_pos,
                                     approach_distance=0.10,
                                     max_steps=100,
                                     threshold=0.01,
                                     approach_axis='z',
                                     timeout=10.0)
            if done:
                print("[Task] Task ended during move to side.")
                return
        except Exception as e:
            print(f"[Error] move() to side position failed: {e}")
            return

        # 3) Move from side to anchor position on drawer
        print(f"[Task] Moving gripper from side to anchor position {anchor_pos}")
        try:
            obs, reward, done = move(env, task,
                                     target_pos=anchor_pos,
                                     approach_distance=0.05,
                                     max_steps=100,
                                     threshold=0.005,
                                     approach_axis='z',
                                     timeout=8.0)
            if done:
                print("[Task] Task ended during move to anchor.")
                return
        except Exception as e:
            print(f"[Error] move() to anchor position failed: {e}")
            return

        # 4) Pick the drawer handle (grasp it)
        print("[Task] Grasping (picking) the drawer handle")
        try:
            obs, reward, done = pick(env, task,
                                     target_pos=anchor_pos,
                                     approach_distance=0.02,
                                     max_steps=80,
                                     threshold=0.005,
                                     approach_axis='z',
                                     timeout=6.0)
            if done:
                print("[Task] Task ended during pick of drawer handle.")
                return
        except Exception as e:
            print(f"[Error] pick() failed: {e}")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        try:
            obs, reward, done = pull(env, task,
                                     target_pos=anchor_pos,
                                     approach_distance=0.0,
                                     max_steps=100,
                                     threshold=0.01,
                                     approach_axis='x',
                                     timeout=10.0)
            if done:
                print("[Task] Drawer pull completed and simulation ended.")
                return
        except Exception as e:
            print(f"[Error] pull() failed: {e}")
            return

        print("[Task] Drawer should now be open. Task complete.")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()