import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if needed
        init_video_writers(obs)

        # Wrap task.step and task.get_observation to record
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Task] Retrieved object positions keys:", list(positions.keys()))

        # Dynamically pick side and anchor position keys to avoid missing-key errors
        side_keys = [k for k in positions.keys() if 'side' in k]
        anchor_keys = [k for k in positions.keys() if 'anchor' in k]
        if not side_keys or not anchor_keys:
            raise KeyError("Could not find 'side' or 'anchor' keys in positions, "
                           f"available keys: {list(positions.keys())}")
        side_pos = positions[side_keys[0]]
        anchor_pos = positions[anchor_keys[0]]
        print(f"[Task] Using side position '{side_keys[0]}' -> {side_pos}")
        print(f"[Task] Using anchor position '{anchor_keys[0]}' -> {anchor_pos}")

        # === Oracle Plan Execution ===

        # 1) Rotate gripper from its default angle to ninety_deg
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, 'zero_deg', 'ninety_deg')
        if done:
            print("[Task] Task ended after rotate")
            return

        # 2) Move gripper to side position
        print(f"[Task] Moving to side position at {side_pos}")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Task ended after move-to-side")
            return

        # 3) Move gripper from side to anchor position
        print(f"[Task] Moving to anchor position at {anchor_pos}")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Task ended after move-to-anchor")
            return

        # 4) Pick up the drawer handle at the anchor position
        print("[Task] Picking drawer handle at anchor position")
        obs, reward, done = pick(env, task, target_pos=anchor_pos,
                                  approach_distance=0.15,
                                  max_steps=100,
                                  threshold=0.01,
                                  approach_axis='z',
                                  timeout=10.0)
        if done:
            print("[Task] Task ended after pick")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended after pull")
            return

        print("[Task] Oracle plan completed successfully!")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()