import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull, move_to_side, move_to_anchor, pick_drawer
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step / get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve a mapping from object names to their 3D positions
        positions = get_object_positions()
        print("[Task] Retrieved positions:", positions)

        # assume the dictionary contains these keys; adjust if your object_positions uses different names
        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']
        # define a 90-degree gripper orientation (axis and values may vary per environment)
        ninety_deg_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])

        # 1) rotate gripper to ninety_deg
        try:
            obs, reward, done = rotate(env, task, ninety_deg_quat)
            if done:
                print("[Task] Terminated during rotate.")
                return
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            return

        # 2) move gripper to drawer side position
        try:
            obs, reward, done = move(env, task, target_pos=drawer_side_pos)
            if done:
                print("[Task] Terminated during move to side.")
                return
        except Exception as e:
            print(f"[Error] move to side failed: {e}")
            return

        # 3) move gripper from side to anchor position
        try:
            obs, reward, done = move(env, task, target_pos=drawer_anchor_pos)
            if done:
                print("[Task] Terminated during move to anchor.")
                return
        except Exception as e:
            print(f"[Error] move to anchor failed: {e}")
            return

        # 4) pick the drawer handle
        try:
            obs, reward, done = pick_drawer(env, task, target_pos=drawer_anchor_pos)
            if done:
                print("[Task] Terminated during pick_drawer.")
                return
        except Exception as e:
            print(f"[Error] pick_drawer failed: {e}")
            return

        # 5) pull the drawer open
        try:
            obs, reward, done = pull(env, task)
            if done:
                print("[Task] Terminated during pull.")
                return
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            return

        print("[Task] Drawer opened successfully.")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()