import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, move_to_side, move_to_anchor, pick_drawer, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Execute a hard‐coded oracle plan using only the provided skills.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset and wrap for recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # fetch all known object positions from the scene
        positions = get_object_positions()
        # assume these keys exist in the scene description
        gripper = 'robot0_gripper'
        drawer = 'drawer0'
        side_pos = positions['drawer0_side_pos']
        anchor_pos = positions['drawer0_anchor_pos']
        target_quat_for_handle = [0.0, 0.0, 0.7071, 0.7071]  # example orientation

        # === 1) Rotate gripper into the correct orientation to approach drawer handle ===
        print("[Task] Rotating into handle‐grasp orientation")
        obs, reward, done = rotate(env, task, target_quat_for_handle)
        if done:
            print("[Task] Early termination after rotate")
            return

        # === 2) Move gripper to side of the drawer ===
        print("[Task] Moving to drawer side position")
        obs, reward, done = move_to_side(env, task, gripper, drawer, side_pos)
        if done:
            print("[Task] Early termination after move_to_side")
            return

        # === 3) Move gripper from side to anchor position (where handle is) ===
        print("[Task] Moving to drawer anchor position")
        obs, reward, done = move_to_anchor(env, task, gripper, drawer, side_pos, anchor_pos)
        if done:
            print("[Task] Early termination after move_to_anchor")
            return

        # === 4) Pick the drawer handle ===
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick_drawer(env, task, gripper, drawer, anchor_pos)
        if done:
            print("[Task] Early termination after pick_drawer")
            return

        # === 5) Pull to open the drawer ===
        print("[Task] Pulling to open drawer")
        obs, reward, done = pull(env, task, gripper, drawer)
        if done:
            print("[Task] Early termination after pull")
            return
        print("[Task] Drawer is now open")

        # === 6) Pick an object from inside the drawer (e.g., tomato0) ===
        obj_name = 'tomato0'
        obj_pos = positions[obj_name]
        print(f"[Task] Picking up {obj_name}")
        obs, reward, done = pick(env, task, obj_name, obj_pos, gripper)
        if done:
            print(f"[Task] Early termination after pick({obj_name})")
            return

        # === 7) Place the object onto its target location (e.g., plate0) ===
        target_name = 'plate0'
        target_pos = positions[target_name]
        print(f"[Task] Placing {obj_name} onto {target_name}")
        obs, reward, done = place(env, task, obj_name, target_pos)
        if done:
            print(f"[Task] Early termination after place({obj_name})")
            return

        print("[Task] Oracle plan completed successfully.")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()