import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step & get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve all object positions from the scene
        positions = get_object_positions()
        print("[Debug] Available position keys:", list(positions.keys()))

        # find side‐position and anchor‐position keys for the drawer handle
        side_keys = [k for k in positions if 'side' in k.lower()]
        anchor_keys = [k for k in positions if 'anchor' in k.lower()]
        if not side_keys or not anchor_keys:
            print("[Error] Could not find side or anchor positions in object_positions().")
            return

        side_pos = positions[side_keys[0]]
        anchor_pos = positions[anchor_keys[0]]

        # step 1: rotate gripper to ninety degrees
        target_quat_ninety = np.array([0.0, 0.0, 0.7071, 0.7071])
        print("[Task] Rotating gripper to 90° orientation.")
        obs, reward, done = rotate(env, task, target_quat_ninety)
        if done:
            print("[Task] Early termination after rotate.")
            return

        # step 2: move to side of drawer
        print(f"[Task] Moving to side position at {side_pos}.")
        obs, reward, done = move(
            env, task,
            target_pos=side_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after move-to-side.")
            return

        # step 3: move to anchor (handle) position
        print(f"[Task] Moving to anchor position at {anchor_pos}.")
        obs, reward, done = move(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after move-to-anchor.")
            return

        # step 4: pick the drawer handle
        print("[Task] Picking up the drawer handle.")
        obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.0,
            max_steps=50,
            threshold=0.005,
            approach_axis='z',
            timeout=5.0
        )
        if done:
            print("[Task] Early termination after pick-drawer.")
            return

        # step 5: pull the drawer open
        print("[Task] Pulling the drawer open.")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Early termination after pull.")
            return

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()