import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve positions of the objects we care about
        positions = get_object_positions()
        # these keys must match what's in object_positions.py
        # e.g., 'tomato', 'plate', 'drawer_side', 'drawer_anchor', 'robot_home', 'ninety_deg_quat'
        tomato_pos = positions['tomato']
        plate_pos = positions['plate']
        drawer_side_pos = positions['drawer_side']
        drawer_anchor_pos = positions['drawer_anchor']
        robot_home_pos = positions.get('robot_home', drawer_side_pos)
        ninety_deg_quat = positions.get('ninety_deg_quat', [0, 0, 0, 1])

        # 1) Move robot to the drawer side to prepare for opening
        print("[Task] Moving from home to drawer side at", drawer_side_pos)
        obs, reward, done = move(env, task, 'robot', robot_home_pos, drawer_side_pos)
        if done:
            print("[Task] Episode ended early after move to drawer side.")
            return

        # 2) Rotate the gripper to the handle orientation (90 degrees)
        print("[Task] Rotating gripper to 90 degrees")
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Task] Episode ended early during rotate.")
            return

        # 3) Move from side position to the anchor (handle) position
        print("[Task] Moving from drawer side to handle anchor at", drawer_anchor_pos)
        obs, reward, done = move(env, task, 'robot', drawer_side_pos, drawer_anchor_pos)
        if done:
            print("[Task] Episode ended early after move to anchor.")
            return

        # 4) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Episode ended early during pull.")
            return

        # 5) Move to the tomato inside the drawer (assumed same position as drawer_anchor)
        print("[Task] Approaching tomato at", tomato_pos)
        obs, reward, done = move(env, task, 'robot', drawer_anchor_pos, tomato_pos)
        if done:
            print("[Task] Episode ended early after moving to tomato.")
            return

        # 6) Pick the tomato
        print("[Task] Picking up the tomato")
        obs, reward, done = pick(env, task, 'tomato', tomato_pos)
        if done:
            print("[Task] Episode ended early during pick.")
            return

        # 7) Move from tomato location to the plate location
        print("[Task] Moving tomato to plate at", plate_pos)
        obs, reward, done = move(env, task, 'robot', tomato_pos, plate_pos)
        if done:
            print("[Task] Episode ended early after moving to plate.")
            return

        # 8) Place the tomato on the plate
        print("[Task] Placing tomato on the plate")
        obs, reward, done = place(env, task, 'tomato', plate_pos)
        if done:
            print("[Task] Episode ended after place.")
            return

        print("===== Task completed successfully =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()