import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import rotate, move, pick, place, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset task and initialize video recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get positions of relevant parts of the drawer
        positions = get_object_positions()
        try:
            side_pos = positions['drawer_side_pos']
            anchor_pos = positions['drawer_anchor_pos']
            handle_pos = positions['drawer_handle']
        except KeyError as e:
            print(f"[Error] Missing object position for key: {e}")
            return

        # 1) Rotate gripper to 90 degrees so it can align with the drawer handle
        print("[Task] Rotating gripper to 90° orientation")
        # quaternion representing a 90° rotation about some axis, for example z
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        obs, reward, done = rotate(env, task, ninety_deg_quat, max_steps=100, threshold=0.02, timeout=5.0)
        if done:
            print("[Task] Task ended during rotate")
            return

        # 2) Move the gripper to the side of the drawer
        print("[Task] Moving to drawer side position:", side_pos)
        obs, reward, done = move(env, task,
                                 target_pos=side_pos,
                                 approach_distance=0.10,
                                 max_steps=100,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=5.0)
        if done:
            print("[Task] Task ended during move to side")
            return

        # 3) Move the gripper to the drawer anchor (the pull handle area)
        print("[Task] Moving to drawer anchor position:", anchor_pos)
        obs, reward, done = move(env, task,
                                 target_pos=anchor_pos,
                                 approach_distance=0.05,
                                 max_steps=100,
                                 threshold=0.005,
                                 approach_axis='z',
                                 timeout=5.0)
        if done:
            print("[Task] Task ended during move to anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Grasping the drawer handle at:", handle_pos)
        obs, reward, done = pick(env, task,
                                 target_pos=handle_pos,
                                 approach_distance=0.02,
                                 max_steps=100,
                                 threshold=0.005,
                                 approach_axis='z',
                                 timeout=5.0)
        if done:
            print("[Task] Task ended during pick")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended during pull")
            return

        print("[Task] Drawer should now be open. Checking final state...")
        # optional: verify is-open predicate via observation if available
    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()