import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We expect positions dict to contain at least:
        #   'drawer_side_pos': np.ndarray([x, y, z])
        #   'drawer_anchor_pos': np.ndarray([x, y, z])
        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            raise KeyError("Missing drawer side or anchor position in get_object_positions()")

        side_pos = positions['drawer_side_pos']
        anchor_pos = positions['drawer_anchor_pos']

        # === Plan Execution ===

        # 1) Rotate gripper to ninety_deg so we can approach the drawer's side handle
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, from_angle='zero_deg', to_angle='ninety_deg')
        if done:
            print("[Task] Episode ended during rotate!")
            return

        # 2) Move to side position of the drawer
        print(f"[Task] Moving to drawer side position at {side_pos}")
        obs, reward, done = move(env, task, target_pos=np.array(side_pos))
        if done:
            print("[Task] Episode ended during move to side!")
            return

        # 3) Move from side to anchor position
        print(f"[Task] Moving to drawer anchor position at {anchor_pos}")
        obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
        if done:
            print("[Task] Episode ended during move to anchor!")
            return

        # 4) Pick the drawer handle (anchor) with the gripper
        print("[Task] Grasping the drawer handle")
        # skill 'pick' expects object, location but here we use pick-drawer
        # we assume pick-drawer is aliased to pick in skill_code
        obs, reward, done = pick(env, task, obj='drawer_handle', room=None, gripper='gripper0')
        if done:
            print("[Task] Episode ended during pick-drawer!")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Episode ended during pull!")
            return

        print("===== Task Completed: Drawer is open =====")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
        raise

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()