import numpy as np
import math
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull, normalize_quaternion
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Exploration: ensure gripper starts at zero_deg orientation ===
        # Define unit quaternions for zero and ninety degrees around z
        zero_deg_quat = np.array([0.0, 0.0, 0.0, 1.0])
        ninety_deg_quat = np.array([0.0, 0.0, math.sin(math.pi/4), math.cos(math.pi/4)])
        # Get current orientation
        obs = task.get_observation()
        current_quat = normalize_quaternion(obs.gripper_pose[3:7])
        # compute angle difference
        dot = np.dot(current_quat, zero_deg_quat)
        if dot < 0:
            dot = -dot
        angle_diff = 2 * math.acos(np.clip(dot, -1.0, 1.0))
        if angle_diff > 0.05:
            print(f"[Exploration] Gripper not at zero_deg (diff={angle_diff:.3f}), rotating to zero_deg.")
            obs, reward, done = rotate(env, task, zero_deg_quat)
            if done:
                print("[Exploration] Task ended during zero_deg rotation.")
                return

        # === Retrieve object positions ===
        positions = get_object_positions()
        # replace these keys with your actual environment keys
        # we assume positions contain 'drawer_side_pos' and 'drawer_anchor_pos'
        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            raise KeyError("Expected keys 'drawer_side_pos' and 'drawer_anchor_pos' in object_positions output")
        side_pos = positions['drawer_side_pos']
        anchor_pos = positions['drawer_anchor_pos']

        # === Oracle Plan Execution ===
        # 1) Rotate gripper to ninety_deg
        print("[Task] Rotating to ninety_deg orientation.")
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Task] Finished during rotate to ninety_deg.")
            return

        # 2) Move to side position of drawer
        print("[Task] Moving to side position:", side_pos)
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Finished during move to side.")
            return

        # 3) Move to anchor position on drawer
        print("[Task] Moving to anchor position:", anchor_pos)
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Finished during move to anchor.")
            return

        # 4) Pick the drawer handle (grasp)
        print("[Task] Picking drawer handle at anchor position.")
        obs, reward, done = pick(env, task, anchor_pos, None)
        if done:
            print("[Task] Finished during pick.")
            return

        # 5) Pull to open the drawer
        print("[Task] Pulling to open the drawer.")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Finished during pull.")
            return

        print("[Task] Drawer should now be open. Plan complete.")

    except Exception as e:
        print(f"[Error] {type(e).__name__}: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()