import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Debug] Retrieved object positions:", positions)

        # Identify keys for drawer, side position, anchor position
        drawer_keys = [k for k in positions.keys() if 'drawer' in k]
        side_keys = [k for k in positions.keys() if 'side' in k]
        anchor_keys = [k for k in positions.keys() if 'anchor' in k]
        if not drawer_keys or not side_keys or not anchor_keys:
            raise RuntimeError("Could not find drawer/side/anchor in positions", positions)

        drawer_name = drawer_keys[0]
        side_pos = positions[side_keys[0]]
        anchor_pos = positions[anchor_keys[0]]
        print(f"[Debug] drawer='{drawer_name}', side_pos={side_pos}, anchor_pos={anchor_pos}")

        # === Task Plan ===

        # 1) Rotate gripper to 90 degrees around some axis
        print("[Plan] Rotating gripper to ninety degrees")
        # A common quaternion for 90° rotation around Y is [0, 0.7071, 0, 0.7071] or similar;
        # adjust according to your frame conventions.
        ninety_deg_quat = np.array([0.0, 0.7071, 0.0, 0.7071])
        try:
            obs, reward, done = rotate(env, task, ninety_deg_quat, max_steps=100, threshold=0.05, timeout=10.0)
        except Exception as e:
            print("[Error] rotate failed:", e)
            return
        if done:
            print("[Task] Ended during rotate")
            return

        # 2) Move to side position of drawer
        print("[Plan] Moving to side position of drawer")
        try:
            obs, reward, done = move(
                env, task,
                target_pos=side_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.02,
                approach_axis='xyz',
                timeout=10.0
            )
        except Exception as e:
            print("[Error] move to side failed:", e)
            return
        if done:
            print("[Task] Ended during move to side")
            return

        # 3) Move to anchor position (handle) of drawer
        print("[Plan] Moving to anchor position of drawer handle")
        try:
            obs, reward, done = move(
                env, task,
                target_pos=anchor_pos,
                approach_distance=0.05,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print("[Error] move to anchor failed:", e)
            return
        if done:
            print("[Task] Ended during move to anchor")
            return

        # 4) Pick (grasp) the drawer handle
        print("[Plan] Grasping drawer handle")
        try:
            obs, reward, done = pick(
                env, task,
                target_pos=anchor_pos,
                approach_distance=0.02,
                max_steps=100,
                threshold=0.005,
                approach_axis='z',
                timeout=5.0
            )
        except Exception as e:
            print("[Error] pick handle failed:", e)
            return
        if done:
            print("[Task] Ended during pick handle")
            return

        # 5) Pull the drawer open
        print("[Plan] Pulling drawer open")
        try:
            obs, reward, done = pull(
                env, task,
                target_pos=anchor_pos,
                max_steps=50,
                threshold=0.01,
                timeout=5.0
            )
        except Exception as e:
            print("[Error] pull drawer failed:", e)
            return
        if done:
            print("[Task] Ended during pull")
            return

        print("[Task] Drawer successfully opened!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()