# run_skeleton_task.py (Fully Generic Skeleton)


import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions is a dict: object_name -> (x,y,z) or similar
        positions = get_object_positions()

        # === Exploration Phase ===
        # We will visit each object or drawer in the scene and use skills
        # to discover missing predicates (identified, temperature-known,
        # weight-known, durability-known, lock-known).
        for name, pos in positions.items():
            try:
                # Move the robot to the location of this entity
                print(f"[Exploration] Moving to '{name}' at position {pos}")
                obs, reward, done = move(env, task, target_pos=pos)
                if done:
                    print("[Exploration] Task ended unexpectedly during move.")
                    return

                # If this is a drawer, attempt to detect its lock status
                if "drawer" in name.lower():
                    print(f"[Exploration] Attempting to pull drawer '{name}' to learn lock state.")
                    # We assume the robot can grasp the drawer handle by a pick
                    obs, reward, done = pick(env, task, target_pos=pos)
                    if done:
                        print("[Exploration] Task ended unexpectedly during pick on drawer.")
                        return
                    # Now pull to learn if it is locked
                    obs, reward, done = pull(env, task, target_pos=pos)
                    if done:
                        print("[Exploration] Task ended unexpectedly during pull.")
                        return
                    print(f"[Exploration] Completed lock-known exploration for '{name}'.")
                else:
                    # Otherwise treat it as a generic object
                    print(f"[Exploration] Picking up object '{name}' to learn weight or durability.")
                    obs, reward, done = pick(env, task, target_pos=pos)
                    if done:
                        print("[Exploration] Task ended unexpectedly during pick on object.")
                        return

                    # Place it back where it was found
                    print(f"[Exploration] Placing object '{name}' back to its original position.")
                    obs, reward, done = place(env, task, target_pos=pos)
                    if done:
                        print("[Exploration] Task ended unexpectedly during place.")
                        return

                    print(f"[Exploration] Completed weight/durability exploration for '{name}'.")
            except Exception as e:
                print(f"[Exploration] Exception encountered with '{name}': {e}")

        # After exploration, you would normally proceed to planning or directly
        # execute the oracle plan using the now-known predicates.
        print("===== Exploration Phase Complete =====")
        # TODO: Insert plan execution here, using only the discovered predicates
        # and the available skills, to achieve the final goal.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
