import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Identify relevant positions from the object list
        bottom_pos = positions['bottom_anchor_pos']
        middle_pos = positions['middle_anchor_pos']
        top_pos = positions['top_anchor_pos']
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']

        # === Execute Task Plan ===

        # Step 1: Rotate the gripper to ninety degrees
        print("[Task] Rotating the gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]))  # Quaternion representation for rotation
        if done:
            print("[Task] Task ended during rotation!")
            return

        # Step 2: Move the gripper to the side position of the middle drawer
        print("[Task] Moving to the side position of the middle drawer.")
        obs, reward, done = move(env, task, target_pos=middle_pos)
        if done:
            print("[Task] Task ended during moving to side position!")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_pos)
        if done:
            print("[Task] Task ended during moving to anchor position!")
            return

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick-drawer(env, task, gripper="gripper", drawer="bottom_drawer", position=bottom_pos)
        if done:
            print("[Task] Task unable to hold the drawer.")
            return

        # Step 5: Pull the bottom drawer fully open
        print("[Task] Pulling the bottom drawer fully open.")
        obs, reward, done = pull(env, task, 0.4, pull_axis='y')  # Adjust pull distance and axis
        if done:
            print("[Task] Task ended during pulling!")
            return

        # Step 6: Pick up the rubbish
        print("[Task] Picking up the rubbish.")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15, max_steps=100, threshold=0.01)
        if done:
            print("[Task] Task ended during picking rubbish!")
            return

        # Step 7: Place the rubbish in the bin
        print("[Task] Placing the rubbish in the bin.")
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, max_steps=100, threshold=0.01)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()