import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import pick, place, pull, rotate, move
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Execute the plan step-by-step according to the specified skills.

        # Step 1: Rotate the gripper to 90 degrees
        print("[Task] Rotating the gripper to 90 degrees...")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi / 4), np.cos(np.pi / 4)]))
        if done:
            print("[Task] Rotated the gripper successfully!")
        else:
            print("[Task] Did not complete rotation!")

        # Step 2: Move to the side position of the bottom drawer
        print("[Task] Moving gripper to side position of the bottom drawer...")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Moved to side position successfully!")
        else:
            print("[Task] Did not complete moving to side position!")

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving gripper to anchor position of the bottom drawer...")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Moved to anchor position successfully!")
        else:
            print("[Task] Did not complete moving to anchor position!")

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer...")
        obs, reward, done = pull(env, task, pull_distance=0.1)  # Adjust the pull_distance based on the simulation scale
        if done:
            print("[Task] Finished pulling open the drawer!")
        else:
            print("[Task] Did not complete pulling the drawer!")

        # Step 5: Pick up any rubbish remaining on the table
        print("[Task] Picking up rubbish from the table...")
        obs, reward, done = pick(env, task, target_pos=positions['rubbish'], approach_distance=0.15)
        if done:
            print("[Task] Finished picking up rubbish!")
        else:
            print("[Task] Did not complete picking up rubbish!")

        # Step 6: Place the rubbish in the bin
        print("[Task] Placing rubbish in the bin...")
        obs, reward, done = place(env, task, target_pos=positions['bin'], approach_distance=0.15)
        if done:
            print("[Task] Finished placing rubbish in the bin!")
        else:
            print("[Task] Did not complete placing rubbish in the bin!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()