import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        
        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Execute the Plan to Achieve Goal ===
        # Step 1: Rotate gripper to desired angle
        print("[Task] Rotating gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05, timeout=10.0)
        
        if done:
            print("[Task] Gripper rotated successfully!")
        else:
            print("[Task] Failed to rotate the gripper.")

        # Step 2: Move to the side position of the bottom drawer
        print("[Task] Moving to side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'], max_steps=100, threshold=0.01, timeout=10.0)
        
        if done:
            print("[Task] Moved to side position of the drawer successfully!")
        else:
            print("[Task] Failed to move to side position of the drawer.")

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving to anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'], max_steps=100, threshold=0.01, timeout=10.0)
        
        if done:
            print("[Task] Moved to anchor position of the drawer successfully!")
        else:
            print("[Task] Failed to move to anchor position of the drawer.")

        # Step 4: Pick the drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, target_pos=positions['bottom_anchor_pos'], approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        
        if done:
            print("[Task] Drawer picked successfully!")
        else:
            print("[Task] Failed to pick the drawer.")

        # Step 5: Pull the drawer open
        print("[Task] Pulling the drawer open.")
        pull_distance = 0.5  # The distance to pull the drawer
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x', max_steps=100, threshold=0.01, timeout=10.0)
        
        if done:
            print("[Task] Drawer pulled open successfully!")
        else:
            print("[Task] Failed to pull the drawer open.")

        # Step 6: Pick up the rubbish
        print("[Task] Picking up the rubbish.")
        obs, reward, done = pick(env, task, target_pos=positions['rubbish'], approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        
        if done:
            print("[Task] Rubbish picked successfully!")
        else:
            print("[Task] Failed to pick the rubbish.")

        # Step 7: Place the rubbish in the bin
        print("[Task] Placing the rubbish in the bin.")
        obs, reward, done = place(env, task, target_pos=positions['bin'], approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        
        if done:
            print("[Task] Rubbish placed in the bin!")
        else:
            print("[Task] Failed to place the rubbish in the bin.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()