import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Define positions
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        rubbish = positions['rubbish']
        bin = positions['bin']
        gripper = "gripper"  # Assuming gripper is a constant for the action calls

        # === Execute the Plan ===
        # Step 1: Rotate the gripper to ensure it can interact with the drawer
        rotate(env, task, target_quat=np.array([0, 0, 0, 1]))  # assuming no rotation needed
        
        # Step 2: Move to the side position next to the bottom drawer
        move(env, task, target_pos=bottom_side_pos)
        
        # Step 3: Move to the anchor position of the bottom drawer
        move(env, task, target_pos=bottom_anchor_pos)
        
        # Step 4: Pick the bottom drawer
        obs, reward, done = pick_drawer(env, task, gripper, "bottom", bottom_anchor_pos)
        if done:
            print("[Task] Ended after picking the drawer!")
            return
        
        # Step 5: Pull the bottom drawer open
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')  # Adjust parameters as needed
        if done:
            print("[Task] Ended after pulling the drawer!")
            return
        
        # Step 6: Pick up the rubbish from the table
        obs, reward, done = pick(env, task, target_pos=rubbish, approach_distance=0.1)
        if done:
            print("[Task] Ended after picking the rubbish!")
            return
        
        # Step 7: Place the rubbish into the bin
        obs, reward, done = place(env, task, target_pos=bin)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()