import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Define positions for objects
        rubbish_pos = positions['rubbish']  # Position of rubbish
        bin_pos = positions['bin']  # Position of bin
        bottom_drawer_pos = positions['bottom_anchor_pos']  # Position where the bottom drawer can be interacted
        side_pos_bottom = positions['bottom_side_pos']  # Position of the bottom side

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Task] Rotating gripper...")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Task ended after rotating the gripper!")
            return

        # Step 2: Move gripper to side position of bottom drawer
        print("[Task] Moving to side position of bottom drawer...")
        obs, reward, done = move(env, task, target_pos=side_pos_bottom, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended after moving to side position!")
            return

        # Step 3: Move gripper to anchor position of bottom drawer
        print("[Task] Moving to anchor position of bottom drawer...")
        obs, reward, done = move(env, task, target_pos=bottom_drawer_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended after moving to anchor position!")
            return

        # Step 4: Pick the bottom drawer
        print("[Task] Picking bottom drawer...")
        obs, reward, done = pick(env, task, target_pos=bottom_drawer_pos)
        if done:
            print("[Task] Task ended after picking the drawer!")
            return

        # Step 5: Pull the bottom drawer to open it
        print("[Task] Pulling bottom drawer...")
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='y', max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Drawer opened successfully!")
            return

        # Step 6: Pick up the rubbish
        print("[Task] Picking up the rubbish...")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended after picking up the rubbish!")
            return

        # Step 7: Place rubbish in the bin
        print("[Task] Placing rubbish in the bin...")
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()