import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Retrieve the necessary positions
        bottom_anchor_pos = positions['bottom_anchor_pos']
        middle_anchor_pos = positions['middle_anchor_pos']
        top_anchor_pos = positions['top_anchor_pos']
        plate_pos = positions['plate']
        
        # Step 1: Rotate gripper for drawer
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100)
        if done:
            print("[Task] Rotation ended successfully!")

        # Step 2: Move to side position of bottom drawer
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos, max_steps=100)
        if done:
            print("[Task] Moved to bottom drawer side position!")

        # Step 3: Move to anchor position of bottom drawer
        obs, reward, done = move(env, task, target_pos=middle_anchor_pos, max_steps=100)
        if done:
            print("[Task] Moved to bottom drawer anchor position!")

        # Step 4: Pick the bottom drawer
        obs, reward, done = pick_drawer(env, task, 'gripper', 'bottom', middle_anchor_pos)
        if done:
            print("[Task] Picked the bottom drawer!")

        # Step 5: Pull open the bottom drawer
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='y', max_steps=100)
        if done:
            print("[Task] Opened the bottom drawer!")

        # Step 6: Pick tomato1 from the table
        obs, reward, done = pick(env, task, 'tomato1', 'table')
        if done:
            print("[Task] Picked tomato1 from the table!")

        # Step 7: Place tomato1 onto the plate
        obs, reward, done = place(env, task, 'tomato1', 'plate')
        if done:
            print("[Task] Placed tomato1 onto the plate!")

        # Step 8: Pick tomato2 from the table
        obs, reward, done = pick(env, task, 'tomato2', 'table')
        if done:
            print("[Task] Picked tomato2 from the table!")

        # Step 9: Place tomato2 onto the plate
        obs, reward, done = place(env, task, 'tomato2', 'plate')
        if done:
            print("[Task] Successfully placed tomato2 onto the plate!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()