import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Step 1: Rotate gripper to the correct position
        try:
            obs, reward, done = rotate(
                env,
                task,
                target_quat=np.array([0, 0, 0, 1]),  # Assuming this is the correct orientation
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            if done:
                print("[Task] Rotation completed successfully!")
        except Exception as e:
            print(f"[Error] {e}")
            return

        # Step 2: Move the gripper to side position to pull the drawer
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=positions['bottom_side_pos'],
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Moved to side position successfully!")
        except Exception as e:
            print(f"[Error] {e}")
            return

        # Step 3: Move to anchor position to pick the drawer
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=positions['bottom_anchor_pos'],
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Moved to anchor position successfully!")
        except Exception as e:
            print(f"[Error] {e}")
            return

        # Step 4: Pull open the drawer if it's not locked
        try:
            obs, reward, done = pull(
                env,
                task,
                pull_distance=0.1,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Drawer pulled successfully!")
            else:
                print("[Task] Failed to pull the drawer.")
                return
        except Exception as e:
            print(f"[Error] {e}")
            return

        # Step 5: Pick tomato1 and drop it onto the plate
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=positions['tomato1'],
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Tomato1 picked successfully!")
            else:
                print("[Task] Failed to pick Tomato1.")
                return
        except Exception as e:
            print(f"[Error] {e}")
            return
        
        # Step 6: Place tomato1 onto the plate
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=positions['plate'],
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Tomato1 placed successfully on the plate!")
            else:
                print("[Task] Failed to place Tomato1.")
                return
        except Exception as e:
            print(f"[Error] {e}")
            return

        # Step 7: Pick tomato2 and drop it onto the plate
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=positions['tomato2'],
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Tomato2 picked successfully!")
            else:
                print("[Task] Failed to pick Tomato2.")
                return
        except Exception as e:
            print(f"[Error] {e}")
            return
        
        # Step 8: Place tomato2 onto the plate
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=positions['plate'],
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Tomato2 placed successfully on the plate!")
            else:
                print("[Task] Failed to place Tomato2.")
                return
        except Exception as e:
            print(f"[Error] {e}")
            return
        
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()