import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_drawer_pos = positions['bottom_anchor_pos']
        plate_pos = positions['plate']

        # === Task Execution Plan ===

        # Step 1: Rotate the gripper to the desired angle
        print("[Task] Rotating the gripper")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Task ended after rotating the gripper!")
            return
        
        # Step 2: Move to the side position near the bottom drawer
        print("[Task] Moving to side position bottom")
        obs, reward, done = move(env, task, bottom_drawer_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended after moving to side position!")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving to anchor position bottom")
        obs, reward, done = move(env, task, bottom_drawer_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended after moving to anchor position!")
            return

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer")
        obs, reward, done = pick_drawer(env, task, "bottom", bottom_drawer_pos)
        if done:
            print("[Task] Task ended after picking the drawer!")
            return

        # Step 5: Pull the bottom drawer open
        print("[Task] Pulling the bottom drawer")
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='y', max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended after pulling the drawer!")
            return

        # Step 6: Pick up tomato1
        print("[Task] Picking up tomato1")
        obs, reward, done = pick(env, task, "tomato1", "table", approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended after picking tomato1!")
            return

        # Step 7: Place tomato1 on the plate
        print("[Task] Placing tomato1 on the plate")
        obs, reward, done = place(env, task, "tomato1", plate_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended after placing tomato1!")
            return

        # Step 8: Pick up tomato2
        print("[Task] Picking up tomato2")
        obs, reward, done = pick(env, task, "tomato2", "table", approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended after picking tomato2!")
            return

        # Step 9: Place tomato2 on the plate
        print("[Task] Placing tomato2 on the plate")
        obs, reward, done = place(env, task, "tomato2", plate_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task completed successfully!")
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()