import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Defining the positions of our items
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        plate_pos = positions['plate']
        bottom_drawer_pos = positions['bottom_side_pos']  # Position to pull drawer
        
        # Step 1: Rotate gripper to 90 degrees for side movement
        print("[Task] Rotating gripper to 90 degrees.")
        obs, reward, done = rotate(env, task, target_quat=[0, 0, 1, 0])  # Adjust quat for 90-degree rotation
        if done:
            print("[Task] Task ended after rotating the gripper!")
            return

        # Step 2: Move gripper to the side position near the bottom drawer
        print("[Task] Moving to side position near the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Task ended after moving to side position!")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving to anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Task ended after moving to anchor position!")
            return

        # Step 4: Pick up the drawer
        print("[Task] Picking up the bottom drawer.")
        obs, reward, done = pick(env, task, target_pos=bottom_drawer_pos)
        if done:
            print("[Task] Task ended after picking the drawer!")
            return

        # Step 5: Pull the drawer
        print("[Task] Pulling the bottom drawer to unlock the cabinet.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')  # Adjust distance as needed
        if done:
            print("[Task] Task ended after pulling the drawer!")
            return

        # Step 6: Pick tomato1 from the table
        print("[Task] Picking tomato1 from the table.")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Task ended after picking tomato1!")
            return

        # Step 7: Place tomato1 onto the plate
        print("[Task] Placing tomato1 onto the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after placing tomato1!")
            return

        # Step 8: Pick tomato2 from the table
        print("[Task] Picking tomato2 from the table.")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Task ended after picking tomato2!")
            return
        
        # Step 9: Place tomato2 onto the plate
        print("[Task] Placing tomato2 onto the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after placing tomato2!")
            return

        # Final output confirming the task completion
        print("[Task] Successfully transferred all tomatoes onto the plate!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()