import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Object positions
        bottom_anchor_pos = positions['bottom_anchor_pos']
        bottom_side_pos = positions['bottom_side_pos']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        plate_pos = positions['plate']

        # === Task Plan Execution ===

        # Step 1: Rotate the gripper to the desired angle (90 degrees)
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Task ended during rotation!")
            return

        # Step 2: Move to the side position of the bottom drawer
        obs, reward, done = move(env, task, bottom_side_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended during movement to side position!")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        obs, reward, done = move(env, task, bottom_anchor_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended during movement to anchor position!")
            return

        # Step 4: Pick the drawer
        obs, reward, done = pick(env, task, 'gripper', 'bottom', 'anchor_pos')
        if done:
            print("[Task] Task ended during picking the drawer!")
            return

        # Step 5: Pull the drawer to open it
        obs, reward, done = pull(env, task, 0.1, 'x', max_steps=100, threshold=0.01, timeout=10.0)  
        if done:
            print("[Task] Task ended during pulling the drawer!")
            return

        # Step 6: Pick tomato 1 from the table
        obs, reward, done = pick(env, task, 'tomato1', 'table')
        if done:
            print("[Task] Task ended during picking tomato 1!")
            return

        # Step 7: Place tomato 1 on the plate
        obs, reward, done = place(env, task, 'tomato1', 'plate')
        if done:
            print("[Task] Task ended during placing tomato 1!")
            return

        # Step 8: Pick tomato 2 from the table
        obs, reward, done = pick(env, task, 'tomato2', 'table')
        if done:
            print("[Task] Task ended during picking tomato 2!")
            return

        # Step 9: Place tomato 2 on the plate
        obs, reward, done = place(env, task, 'tomato2', 'plate')
        if done:
            print("[Task] Task ended during placing tomato 2!")
            return

        print("[Task] Task completed successfully!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()