import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *  # Importing existing skill functions
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = {
            'tomato1': (0.5, 0.0, 0.0),
            'tomato2': (0.5, 0.1, 0.0),
            'plate': (0.6, 0.0, 0.0),
            'bottom': (0.0, 0.0, 0.0),
            'side-pos-bottom': (0.0, 0.1, 0.0),
            'anchor-pos-bottom': (0.0, 0.2, 0.0)
        }

        # Execute the plan step-by-step
        print("Executing the oracle plan...")

        # Step 1: Rotate gripper to 90 degrees
        obs, reward, done = rotate(env, task, target_quat=np.array([0.0, 0.0, 0.7071, 0.7071]))  
        if done: return

        # Step 2: Move to side position for bottom drawer
        obs, reward, done = move(env, task, target_pos=positions['side-pos-bottom'])  
        if done: return

        # Step 3: Move to anchor position for bottom drawer
        obs, reward, done = move(env, task, target_pos=positions['anchor-pos-bottom'])  
        if done: return
        
        # Step 4: Pick the bottom drawer
        obs, reward, done = pick(env, task, obj='bottom', loc=positions['anchor-pos-bottom'])  
        if done: return
        
        # Step 5: Pull the bottom drawer to open it
        obs, reward, done = pull(env, task, pull_distance=0.5)  
        if done: return
        
        # Step 6: Pick first tomato using gripper
        obs, reward, done = pick(env, task, obj='tomato1', loc=positions['tomato1'])  
        if done: return
        
        # Step 7: Place first tomato on plate
        obs, reward, done = place(env, task, obj='tomato1', loc=positions['plate'])  
        if done: return

        # Step 8: Pick second tomato using gripper
        obs, reward, done = pick(env, task, obj='tomato2', loc=positions['tomato2'])  
        if done: return
        
        # Step 9: Place second tomato on plate
        obs, reward, done = place(env, task, obj='tomato2', loc=positions['plate'])  
        if done: return

        print("All steps executed successfully!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()