import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
        bottom_side_pos = positions.get('bottom_side_pos', None)
        tomato1_pos = positions.get('tomato1', None)
        tomato2_pos = positions.get('tomato2', None)
        plate_pos = positions.get('plate', None)

        # Check that the necessary positions have been retrieved
        if any(v is None for v in [bottom_anchor_pos, bottom_side_pos, tomato1_pos, tomato2_pos, plate_pos]):
            print("Error: One or more object positions could not be found.")
            return

        # === Task Plan Execution ===

        # Step 1: Rotate the gripper to the desired angle (90 degrees)
        print("Step 1: Rotate gripper to 90 degrees")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, np.sqrt(0.5), np.sqrt(0.5)]))  
        if done:
            print("[Step 1] Task ended during rotation.")
            return

        # Step 2: Move the gripper to the side position of the bottom drawer
        print("Step 2: Move to the side position of the drawer")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Step 2] Task ended during move.")
            return

        # Step 3: Move the gripper to the anchor position of the bottom drawer
        print("Step 3: Move to the anchor position of the drawer")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Step 3] Task ended during move.")
            return

        # Step 4: Pick the drawer
        print("Step 4: Pick the drawer")
        obs, reward, done = pick_drawer(env, task, gripper='gripper', drawer='bottom', position=bottom_anchor_pos)
        if done:
            print("[Step 4] Task ended during picking drawer.")
            return

        # Step 5: Pull the drawer open
        print("Step 5: Pull the drawer")
        obs, reward, done = pull(env, task, pull_distance=0.15)  
        if done:
            print("[Step 5] Task ended during pulling drawer.")
            return
        
        # Step 6: Pick tomato1 from the table
        print("Step 6: Pick tomato1")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Step 6] Task ended during picking tomato1.")
            return

        # Step 7: Place tomato1 on the plate
        print("Step 7: Place tomato1 on the plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Step 7] Task ended during placing tomato1.")
            return

        # Step 8: Pick tomato2 from the table
        print("Step 8: Pick tomato2")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Step 8] Task ended during picking tomato2.")
            return

        # Step 9: Place tomato2 on the plate
        print("Step 9: Place tomato2 on the plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Step 9] Task ended during placing tomato2.")
            return

        print("===== Task Completed Successfully =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()