import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *  # Import existing skills like pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Define positions for convenience
        bottom_anchor_pos = positions['bottom_anchor_pos']
        bottom_side_pos = positions['bottom_side_pos']
        plate_pos = positions['plate']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']

        # --- Step-by-Step Execution Based on the Plan ---
        
        # Step 1: Rotate the gripper to Ninety Degrees
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]))  # Rotate to 90 degrees
        if done:
            print("[Task] Task ended after rotation!")
            return
        
        # Step 2: Move to Side Position of Bottom Drawer
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Task ended after moving to side position!")
            return
        
        # Step 3: Move to Anchor Position of Bottom Drawer
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Task ended after moving to anchor position!")
            return
        
        # Step 4: Pick the Bottom Drawer
        obs, reward, done = pick(env, task, 'gripper', 'bottom', 'anchor-pos-bottom')
        if done:
            print("[Task] Task ended after picking the drawer!")
            return
        
        # Step 5: Pull the drawer to open it
        obs, reward, done = pull(env, task, pull_distance=0.1)
        if done:
            print("[Task] Task ended after pulling the drawer!")
            return
        
        # Step 6: Pick the first tomato
        obs, reward, done = pick(env, task, 'tomato1', 'table')  
        if done:
            print("[Task] Task ended after picking tomato1!")
            return
        
        # Step 7: Place the first tomato on the plate
        obs, reward, done = place(env, task, 'tomato1', 'plate')
        if done:
            print("[Task] Task ended after placing tomato1!")
            return
        
        # Step 8: Pick the second tomato
        obs, reward, done = pick(env, task, 'tomato2', 'table')
        if done:
            print("[Task] Task ended after picking tomato2!")
            return
        
        # Step 9: Place the second tomato on the plate
        obs, reward, done = place(env, task, 'tomato2', 'plate')
        if done:
            print("[Task] Task completed successfully with both tomatoes placed!")
            return
        
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()