import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions
        top_drawer_pos = positions['top_anchor_pos']
        plate_pos = positions['plate']

        # === Execute the Plan ===
        # Step 1: Pull the top drawer
        print("[Task] Pulling the top drawer...")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y', max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended after pulling the drawer!")
            return

        # Step 2: Rotate the gripper to the correct orientation
        print("[Task] Rotating the gripper to ninety degrees...")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Task ended after rotating the gripper!")
            return

        # Step 3: Pick tomato1 from the table
        print("[Task] Picking tomato1 from the table...")
        obs, reward, done = pick(env, task, target_pos=positions['item1'], approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended after picking tomato1!")
            return

        # Step 4: Place tomato1 on the plate
        print("[Task] Placing tomato1 on the plate...")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended after placing tomato1!")
            return

        # Step 5: Pick tomato2 from the table
        print("[Task] Picking tomato2 from the table...")
        obs, reward, done = pick(env, task, target_pos=positions['item2'], approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended after picking tomato2!")
            return

        # Step 6: Place tomato2 on the plate
        print("[Task] Placing tomato2 on the plate...")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()