def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Assuming the 'gripper' is initialized/available in some predefined way
        # Assuming positions includes 'room_a', 'room_b', 'drawer_d', etc.
        room_a = positions['room_a']
        room_b = positions['room_b']
        drawer_d = positions['drawer_d']
        gripper = positions.get('gripper')  # Ensure gripper is accounted

        if gripper is None:
            print("[Error] Gripper object not found in the environment. Exiting.")
            return

        print("[Task] Moving to room A for initial action.")
        obs, reward, done = move(env, room_a, room_b)

        if done:
            print("[Task] Task ended after moving to room A!")
            return

        print("[Task] Picking an object in room A.")
        obs, reward, done = pick(env, "object_1", room_a, gripper)

        if done:
            print("[Task] Task ended after picking object in room A!")
            return

        print("[Task] Placing the object in room B.")
        obs, reward, done = place(env, "object_1", room_b)

        if done:
            print("[Task] Task ended after placing object in room B!")
            return

        print("[Task] Rotating gripper to adjust position.")
        target_quat = [0, 0, 0, 1]  # replace with the correct orientation
        obs, reward, done = rotate(env, task, target_quat)

        if done:
            print("[Task] Task ended after rotating gripper!")
            return

        print("[Task] Opening the drawer.")
        obs, reward, done = move(env, gripper, drawer_d, "side_position")

        if done:
            print("[Task] Task ended with opening the drawer!")
            return
        
        print("[Task] Picking drawer handle.")
        obs, reward, done = pick(env, "drawer_handle", drawer_d, gripper)
        
        if done:
            print("[Task] Task finished after picking the drawer handle!")
            return

        print("[Task] Pulling the drawer open.")
        obs, reward, done = pull(env, "drawer_handle", drawer_d)

        if done:
            print("[Task] Task finished after pulling the drawer!")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()