import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions based on the provided object list
        bottom_drawer_pos = positions['bottom_anchor_pos']
        bin_pos = positions['bin']
        rubbish_pos = positions['item3']  # Assuming item3 is the rubbish

        # === Execute the Plan ===
        # Step 1: Rotate the gripper to ninety degrees
        try:
            obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05, timeout=10.0)
            if done:
                print("[Task] Gripper rotated successfully!")
            else:
                print("[Task] Failed to rotate the gripper.")
        except Exception as e:
            print(f"[Error] {e}")

        # Step 2: Move to the side position of the bottom drawer
        try:
            obs, reward, done = move(env, task, target_pos=bottom_drawer_pos, max_steps=100, threshold=0.01, timeout=10.0)
            if done:
                print("[Task] Moved to the side position of the bottom drawer successfully!")
            else:
                print("[Task] Failed to move to the side position.")
        except Exception as e:
            print(f"[Error] {e}")

        # Step 3: Move to the anchor position of the bottom drawer
        try:
            obs, reward, done = move(env, task, target_pos=bottom_drawer_pos, max_steps=100, threshold=0.01, timeout=10.0)
            if done:
                print("[Task] Moved to the anchor position of the bottom drawer successfully!")
            else:
                print("[Task] Failed to move to the anchor position.")
        except Exception as e:
            print(f"[Error] {e}")

        # Step 4: Pick the bottom drawer
        try:
            obs, reward, done = pick(env, task, target_pos=bottom_drawer_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
            if done:
                print("[Task] Picked the bottom drawer successfully!")
            else:
                print("[Task] Failed to pick the bottom drawer.")
        except Exception as e:
            print(f"[Error] {e}")

        # Step 5: Pull open the bottom drawer
        try:
            obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y', max_steps=100, threshold=0.01, timeout=10.0)
            if done:
                print("[Task] Drawer pulled successfully!")
            else:
                print("[Task] Failed to pull the drawer.")
        except Exception as e:
            print(f"[Error] {e}")

        # Step 6: Pick the rubbish
        try:
            obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
            if done:
                print("[Task] Picked the rubbish successfully!")
            else:
                print("[Task] Failed to pick the rubbish.")
        except Exception as e:
            print(f"[Error] {e}")

        # Step 7: Place the rubbish into the bin
        try:
            obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
            if done:
                print("[Task] Rubbish placed in the bin successfully!")
            else:
                print("[Task] Failed to place the rubbish in the bin.")
        except Exception as e:
            print(f"[Error] {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()