import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions
        rubbish_pos = positions['item3']  # Assuming item3 is the rubbish
        bin_pos = positions['bin']  # Assuming bin is the target for rubbish
        bottom_drawer_pos = positions['bottom_anchor_pos']  # Assuming this is the position of the bottom drawer

        # Step 1: Rotate the gripper to the correct orientation
        print("[Task] Rotating the gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]))  # Rotate to 90 degrees
        if done:
            print("[Task] Task ended after rotating the gripper!")
            return

        # Step 2: Move to the side position of the bottom drawer
        print("[Task] Moving to the side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_drawer_pos + np.array([0, 0, 0.1]))  # Adjust height slightly
        if done:
            print("[Task] Task ended after moving to the side position!")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_drawer_pos)
        if done:
            print("[Task] Task ended after moving to the anchor position!")
            return

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, target_pos=bottom_drawer_pos, approach_distance=0.15)
        if done:
            print("[Task] Task ended after picking the drawer!")
            return

        # Step 5: Pull the bottom drawer open
        print("[Task] Pulling the bottom drawer open.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y', max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended after pulling the drawer!")
            return

        # Step 6: Pick up the rubbish
        print("[Task] Picking up the rubbish.")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended after picking up the rubbish!")
            return

        # Step 7: Place the rubbish in the bin
        print("[Task] Placing rubbish in the bin.")
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()