import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *  # Importing predefined skills
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define positions for the drawers and the bin
        top_drawer_pos = positions['top_anchor_pos']
        middle_drawer_pos = positions['middle_anchor_pos']
        bottom_drawer_pos = positions['bottom_anchor_pos']
        bin_pos = positions['bin']

        # === Execute the Plan ===
        # Step 1: Rotate the gripper to the required angle
        print("[Task] Rotating the gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100)
        if done:
            print("[Task] Rotation completed.")
        
        # Step 2: Move to the side position of the bottom drawer
        print("[Task] Moving to the side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Moved to side position.")

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Moved to anchor position.")

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, target_pos=bottom_drawer_pos)
        if done:
            print("[Task] Picked the bottom drawer.")

        # Step 5: Pull open the bottom drawer
        print("[Task] Pulling open the bottom drawer.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y')
        if done:
            print("[Task] Pulled open the bottom drawer.")

        # Step 6: Pick the rubbish from the table
        print("[Task] Picking the rubbish from the table.")
        obs, reward, done = pick(env, task, target_pos=positions['item1'])  # Assuming rubbish is item1
        if done:
            print("[Task] Picked the rubbish.")

        # Step 7: Place the rubbish into the bin
        print("[Task] Placing the rubbish into the bin.")
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()